/*
 * Typical pin layout used:
 * ---------------------------------------------------------
 *               NSP32      Arduino       Arduino   Arduino      ESP32
 *  SPI                     Uno/101       Mega      Nano v3 
 * Signal        Pin          Pin           Pin       Pin     
 * -----------------------------------------------------------
 * Wakeup/Reset  RST          8             49        D8           27
 * SPI SSEL      SS           10            53        D10          5
 * SPI MOSI      MOSI         11 / ICSP-4   51        D11          23
 * SPI MISO      MISO         12 / ICSP-1   50        D12          19
 * SPI SCK       SCK          13 / ICSP-3   52        D13          18
 * Ready         Ready        2             21        D2           17
 *
 */

#include "ArduinoAdaptor.h"
#include "NSP32.h"
#include <SPI.h>
#include "FS.h"
#include <TFT_eSPI.h> // Hardware-specific library
#include <ESP32Time.h>
#include <Wire.h>
#include "SD.h"

// #include "EloquentTinyML.h" 
// #include "eloquent_tinyml/tensorflow.h" 
// #include "model.h" 


ESP32Time rtc;
/***********************************************************************************
 *  ili9431 setting                                                     *
 ***********************************************************************************/
TFT_eSPI tft = TFT_eSPI(); // Invoke custom library

// This is the file name used to store the calibration data
// You can change this to create new calibration files.
// The SPIFFS file name must start with "/".
#define CALIBRATION_FILE "/TouchCalData1"

// Set REPEAT_CAL to true instead of false to run calibration
// again, otherwise it will only be done once.
// Repeat calibration if you change the screen rotation.
#define REPEAT_CAL false

#define KEY_X 40 //按鍵位置
#define KEY_Y 200  //按鍵位置
#define KEY_W 72  //按鍵寬度
#define KEY_H 32  //按鍵高度
#define KEY_SPACING_X 5 // 按鍵間距
#define KEY_SPACING_Y 20 // 按鍵間距
#define KEY_TEXTSIZE  1 // Font size multiplier

// Using two fonts since numbers are nice when bold
#define LABEL1_FONT &FreeSansOblique12pt7b // Key label font 1
#define LABEL2_FONT &FreeSansBold12pt7b    // Key label font 2

#define DISP_X 0 //示波畫面左上角X位置
#define DISP_Y 20 //示波畫面左上角y位置
#define DISP_W 240 //示波畫面寬度
#define DISP_H 150 //示波畫面高度

// Create 3 keys for the keypad
char keyLabel[3][7] = {"Start", "PPG", "Reset"};
uint16_t keyColor[3] = {TFT_DARKGREY, TFT_DARKGREY, TFT_DARKGREY};

// Invoke the TFT_eSPI button class and create all the button objects
TFT_eSPI_Button key[3];

String lines[150] ={};

/***********************************************************************************
 *  TCA6507 LED Driver setting                                                     *
 ***********************************************************************************/
#define BUFFER_LENGTH 32
#define SDA 21
#define SCL 22
#define EN 26
#define device_address 0x45
enum TCA_registers{
  SELECT0 = 0x00,
  SELECT1 = 0x01,
  SELECT2 = 0x02,
  FADE_ON_TIME = 0x03,
  FULLY_ON_TIME = 0x04,
  FADE_OFF_TIME = 0x05,
  FIRST_FULLY_OFF_TIME = 0x06,
  SECOND_FULLY_OFF_TIME = 0x06,
  MAXIMUM_INTENSITY = 0x08,
  ONE_SHOT = 0x09,
  INITIALIZATION = 0x10,
};

/***********************************************************************************
 *  NSP32 pins and setting                                                         *
 ***********************************************************************************/

const unsigned int PinRst = 27;    // pin Reset
const unsigned int PinReady = 17;  // pin Ready

using namespace NanoLambdaNSP32;

ArduinoAdaptor adaptor(PinRst);            // master MCU adaptor
NSP32 nsp32(&adaptor, NSP32::ChannelSpi);  // NSP32 (using SPI channel)

int NOP = 0; // number of points
/***********************************************************************************
 *  SD SSPin and setting                                                           *
 ***********************************************************************************/
#define SD_CS_PIN 25
SPIClass SDspi = SPIClass(HSPI);
unsigned long startTime ;//開始計時時間
unsigned long Time;
const char fileformat[5]=".csv"; //檔案格式
bool SDconnect=false;

/***********************************************************************************
 *  all Function                                                                   *
 ***********************************************************************************/
void PinReadyTriggerISR();
void drawKeypad();
void touch_calibrate();
int8_t TCA_writeBytes(uint8_t devAddr, uint8_t regAddr, uint8_t length, uint8_t *data);
int8_t TCA_writeByte(uint8_t devAddr, uint8_t regAddr, uint8_t data);
int8_t TCA_read(uint8_t devAddr, uint8_t regAddr, uint8_t length, uint8_t *data, uint16_t timeout);
bool TCA_isConnected();
void listDir(fs::FS &fs, const char * dirname, uint8_t levels);
void createDir(fs::FS &fs, const char * path);
void removeDir(fs::FS &fs, const char * path);
void readFile(fs::FS &fs, String path);
void writeFile(fs::FS &fs, String path, float message[] , int messageSize);
void appendFileAWPPG(fs::FS &fs, String path, float message[] , int messageSize);
void appendFilePPG(fs::FS &fs, String path, float message[] , int messageSize);
void appendFilePPGtimes(fs::FS &fs, String path, unsigned long message[] , int messageSize);
void renameFile(fs::FS &fs, String path1, String path2);
void deleteFile(fs::FS &fs, String path);
void testFileIO(fs::FS &fs, String path);
int filenumber();
void displayWaveform(float Wave[], int timePoints, int screen_x, int screen_y, int screen_w, int screen_h);

/***********************************************************************************
 *  Tensorflow lite setting                                                        *
 ***********************************************************************************/
// //Tensorflow lite
// #define NUMBER_OF_INPUTS  200
// #define NUMBER_OF_OUTPUTS 2
// #define TENSOR_ARENA_SIZE 4 * 1024 // 模型使用記憶體大小
// Eloquent::TinyML::TensorFlow::MutableTensorFlow <NUMBER_OF_INPUTS, NUMBER_OF_OUTPUTS, TENSOR_ARENA_SIZE> tf;

//------------------------------------------------------------------------------------------
//SETUP

void setup() {
  Serial.begin(115200);

  // Use serial port
  pinMode(TFT_CS, OUTPUT);
  pinMode(TOUCH_CS, OUTPUT);
  pinMode(SD_CS_PIN,OUTPUT);

  Wire.begin(SDA, SCL);

  pinMode(EN, OUTPUT);
  digitalWrite(EN, LOW);
  delay(20);
  digitalWrite(EN, HIGH);
  delay(20);


  if(TCA_isConnected()){
    Serial.println("TCA6507 connected!!");
  }
  else{
    Serial.println("TCA6507 doesn't connected.");
  }
  
  // Initialise the TFT screen
  tft.init();

  // Set the rotation before we calibrate
  tft.setRotation(2);

  // Calibrate the touch screen and retrieve the scaling factors
  //touch_calibrate();
  uint16_t calData[5] = { 336, 3250, 446, 3404, 2 };
  tft.setTouch(calData);

  // Clear the screen
  tft.fillScreen(TFT_BLACK);

  // Draw keypad
  drawKeypad();
  //顯示文字
  tft.setTextSize(1);
  tft.setTextColor(TFT_WHITE, TFT_BLACK);
  tft.setTextPadding(75);
  tft.drawString("Spectrogram", 5, 0, 2);

  tft.setTextSize(1);
  tft.setTextColor(TFT_WHITE, TFT_BLACK);
  tft.drawString("RESULT :", 5, 265, 4);

  
  //示波視窗
  tft.fillRect(DISP_X, DISP_Y, DISP_W, DISP_H, TFT_BLACK);
  tft.drawRect(DISP_X, DISP_Y, DISP_W, DISP_H, TFT_WHITE);


  // initialize "ready trigger" pin for accepting external interrupt (falling edge trigger)
  pinMode(PinReady, INPUT_PULLUP);                                                // use pull-up
  attachInterrupt(digitalPinToInterrupt(PinReady), PinReadyTriggerISR, FALLING);  // enable interrupt for falling edge


  rtc.setTime(0, 0, 12, 10, 7, 2024);

	// initialize NSP32
	nsp32.Init();
  Serial.println("Init");
  
	// =============== standby ===============
	nsp32.Standby(0);
  Serial.println("Standby");
	// =============== wakeup ===============
	nsp32.Wakeup();
  Serial.println("Wakeup");
	// =============== hello ===============
	nsp32.Hello(0);
  Serial.println("Hello");
	// =============== get sensor id ===============
	nsp32.GetSensorId(0);
  Serial.println("get sensor id");
  Serial.println();

  char szSensorId[15];                   // sensor id string buffer
  nsp32.ExtractSensorIdStr(szSensorId);  // now we have sensor id string in szSensorId[]

  Serial.print(F("sensor id = "));
  Serial.println(szSensorId);

  // =============== get wavelength ===============
  nsp32.GetWavelength(0);

  WavelengthInfo infoW;
  nsp32.ExtractWavelengthInfo(&infoW);  // now we have all wavelength info in infoW, we can use e.g. "infoW.Wavelength" to access the wavelength data array
  
  NOP = infoW.NumOfPoints;

  Serial.printf("Number of wavelength = %d\n", NOP);

  Serial.println(F("Elements of wavelength = "));
  for(int i = 0; i < NOP; i++){
    Serial.printf("%d ",infoW.Wavelength[i]);
  }
  Serial.print("\n");

  
  //SD initialize

  if(!SD.begin(SD_CS_PIN,SDspi,100000000)){
    Serial.println("SD Card Failed");
    tft.setTextSize(1);
    tft.setTextColor(TFT_RED, TFT_BLACK);
    tft.drawString("SD Failed", 5, 230, 4);
  }
  else{
    Serial.println("SD Card initialize done");
    tft.setTextSize(1);
    tft.setTextColor(TFT_GREEN, TFT_BLACK);
    tft.drawString("SD connected", 5, 230, 4);
    SDconnect=true;
  }

  // //model initialize
  // tf.addBuiltinOp(BuiltinOperator_FULLY_CONNECTED, Register_FULLY_CONNECTED(), 1, 9);
  // tf.addBuiltinOp(BuiltinOperator_SOFTMAX, Register_SOFTMAX(), 1, 1);
  // tf.addBuiltinOp(BuiltinOperator_LOGISTIC, Register_LOGISTIC(), 1, 1);
  // tf.begin((unsigned char*) model_tflite);
}


//---------------------------------------------------------------------------------------------------------------
//LOOP

int start_pause = 0;//光譜按鍵計數
int reset = 0;//reset按鍵
int PPGdetect = 0;//模型預測按鍵計數
const int times=200;//橫軸光譜資料數
int t=0;//取times次光譜資料計數
String fileName = "/AWPPG" + String(0) + fileformat;//檔名
float PPG[times] = {0}; //全部加起來的PPG訊號
unsigned long PPGtimes[times] = {0}; //PPG時間軸


void loop() {
  int timer = rtc.getSecond();

  tft.setTextPadding(0);
  uint16_t t_x = 0, t_y = 0; // To store the touch coordinates

  // Pressed will be set true is there is a valid touch on the screen
  bool pressed = tft.getTouch(&t_x, &t_y);

  // / Check if any key coordinate boxes contain the touch coordinates
  //可按按鍵數
  for (uint8_t b = 0; b < 3; b++) {
    if (pressed && key[b].contains(t_x, t_y)) {
      key[b].press(true);  // tell the button it is pressed
    } else {
      key[b].press(false);  // tell the button it is NOT pressed
    }
  }
  // Check if any key has changed state
  for (uint8_t b = 0; b < 3; b++) {
    tft.setFreeFont(LABEL1_FONT);
    if (key[b].justReleased()) key[b].drawButton();     // draw normal
    if (key[b].justPressed()) {
      key[b].drawButton(true);  // draw invert

    // Del button, so delete last char 按鍵功能
      if (b == 0) {
        Serial.println("\nSTART");
        start_pause++;
      }
      if (b == 1) {
        Serial.println("\nPPGDETECT");
        PPGdetect++;
      }
      if (b == 2) {
        Serial.println("\nRESET");
        reset++;
      }

      //delay(10); // UI debouncing
    }
  }
  
//按鍵動作
  float std_spec[NOP] = {0}; 
  
//畫光譜----------------------------------------------------------------------------------------------
  if(start_pause % 2 == 1){
    if (key[0].justReleased()) key[0].drawButton(false,String("Stop"));     // draw normal

    uint8_t selects[3] = {0x00, 0x00, 0x1E};
    TCA_writeByte(device_address, SELECT2, selects[2]); //開燈

    //重刷***********************************************************
    if (t==0){
      delay(500); //開燈延遲半秒後開始畫光譜
      for (int i = 0; i < times; i++){ //重刷PPG資料
        PPG[i] = 0;
      }
      tft.fillRect(DISP_X + 1, DISP_Y + 1, DISP_W - 2, DISP_H - 2, TFT_BLACK);
      tft.drawRect(DISP_X, DISP_Y, DISP_W, DISP_H, TFT_WHITE);
      tft.setTextSize(1);
      tft.setTextColor(TFT_YELLOW, TFT_BLACK);
      tft.setTextPadding(200); 
      tft.drawString("times : counting", 130, 2, 2);  //印出計時中
      startTime = millis();  //開始計時
    }
    Time = millis()-startTime;

    // ========== spectrum acquisition ===========
    nsp32.AcqSpectrum(0, 20, 1, false); // integration time = 20; exposure time =36ms; frame avg num = 1; disable AE

    //"AcqSpectrum" command takes longer time to execute, the return packet is not immediately available
    //when the acquisition is done, a "ready trigger" will fire, and nsp32.GetReturnPacketSize() will be > 0
    while (nsp32.GetReturnPacketSize() <= 0) {
      // TODO: can go to sleep, and wakeup when "ready trigger" interrupt occurs
      nsp32.UpdateStatus();  // call UpdateStatus() to check async result
    }

    SpectrumInfo infoS;
    nsp32.ExtractSpectrumInfo(&infoS);  // now we have all spectrum info in infoW, we can use e.g. "infoS.Spectrum" to access the spectrum data array
    
    //取資料與正規化***************************************************************
    float maxValue = 0;
    float minValue = 0;
    for(int i = 0; i < NOP; i++) {
      maxValue = max(maxValue, infoS.Spectrum[i]);
      minValue = min(minValue, infoS.Spectrum[i]);
    }
    float d = maxValue - minValue;
    for (int i = 0; i < NOP; i++) {
      std_spec[i] = (infoS.Spectrum[i]-minValue)/d;
      PPG[t] += infoS.Spectrum[i];
      //Serial.printf("%.6f ", std_spec[i]);
      //timeStepData[t][i] = std_spec[i]; // 從正規化後陣列中獲取當次資料
    }

    //save data to SD card*************************************************************************
    if (t==0 & SDconnect){
      fileName = "/AWPPG" + String(filenumber()) + fileformat;
      if (!SD.exists(fileName)) {
        tft.setTextSize(1);
        tft.setTextColor(TFT_GREEN, TFT_BLACK);
        tft.setTextPadding(200); 
        tft.drawString("save:" + fileName, 5, 230, 4);
      }
    }
    Serial.printf("%d ",t);
    appendFileAWPPG(SD, fileName, infoS.Spectrum, NOP);
    PPGtimes[t] = Time;
    //Serial.println(nsp32.m_retBuf[50]);

    
    //計數
    t++;  //從1開始算到times
    if (t>=times) { 
      t=0;
      //readFile(SD, fileName);
      start_pause++;
      key[0].drawButton();

      tft.setTextSize(1);
      tft.setTextColor(TFT_YELLOW, TFT_BLACK);
      tft.setTextPadding(200); 
      tft.drawString("times : " + String(Time)+ "ms", 130, 2, 2);  //印出計時時間
      
      fileName = "/PPG" + String(filenumber() - 1) + fileformat;
      appendFilePPG(SD, fileName, PPG, times);//存PPG單波長訊號
      appendFilePPGtimes(SD, fileName, PPGtimes, times);//存PPG單波長訊號
    }

    // 繪製時頻譜圖示波**************************************************************
    tft.setTextSize(1);
    tft.setTextColor(TFT_WHITE, TFT_BLACK);
    tft.setTextPadding(75);
    tft.drawString("Spectrogram", 5, 0, 2);
    int color;
    int red;
    int green;
    int blue;

    for (int j = 0; j < NOP; j++) {
      float normalizedVal = std_spec[j];//timeStepData[t][j];//
      if (normalizedVal >= 0.83 && normalizedVal <= 1) {
        //紅色
        red=255;
        green=0;
        blue=0;
      }
      else if(normalizedVal >= 0.66 && normalizedVal < 0.83) {
        //橘色
        red=255;
        green=165;
        blue=0;
      }
      else if(normalizedVal >= 0.5 && normalizedVal < 0.66) {
        //黃色
        red=255;
        green=255;
        blue=0;
      }
      else if(normalizedVal >= 0.33 && normalizedVal < 0.5) {
        //綠色
        red=0;
        green=255;
        blue=0;
      }
      else if(normalizedVal >= 0.16 && normalizedVal < 0.33) {
        //藍色
        red=0;
        green=0;
        blue=255;
      }
      else {
        //黑色
        red=0;
        green=0;
        blue=0;
      }
      color = tft.color565(red, green, blue);
      // 繪製像素
      int x = (DISP_X+1 + t) ;//時間軸(橫軸)
      int y = (DISP_Y+DISP_H-2 -j) ;//頻率軸(縱軸)
      tft.drawPixel(x, y, color);
    }
    //計次與時間***********************************************************************
    //counter++;
    //Serial.printf("count: %d\n", counter);
    //Serial.print(rtc.getTime("%A, %B %d %Y %H:%M:%S:"));
    //Serial.println(rtc.getMillis());
  }
  else{
    if (t>0){
      tft.setTextSize(1);
      tft.setTextColor(TFT_YELLOW, TFT_BLACK);
      tft.setTextPadding(200); 
      tft.drawString("times : " + String(millis() - startTime)+ "ms", 130, 2, 2);  //印出計時時間
    }
    
    t=0;
    TCA_writeByte(device_address, SELECT2, 0x00); //關燈
  }

//模型預測----------------------------------------------------------------------------------------------
  if(PPGdetect % 2 == 1 && t==0){
    // float test_list[25] = {0.0, 0.003030717, 0.018056406, 0.007386915, 0.0, 0.001869699, 0.000449, 0.000718, 0.006982542, 0.016715601, 0.024452891, 0.028154038, 0.047208875, 0.047339678, 0.033032645, 0.029521797, 0.03147052, 0.026025888, 0.033922244, 0.060511492, 0.07208954, 0.056529667, 0.035075348, 0.06048418, 0.022952799};
    // Serial.printf("Std features = ");
    // for(int i = 0; i < 75; i++){
    //   if(i % 3 == 0){
    //     std_features[i / 3] = (float) std_spec[i];
    //     Serial.printf("%.6f ", std_features[i / 3]);
    //   }
    // }

    // for(int i = 0; i < times; i++){
    //   Serial.printf("%.6f ", PPG[i]);
    // }
    // float prediction[2] = {};
    // //tf.predict(test_list, prediction);
    // tf.predict(PPG, prediction);
    // Serial.printf("\nReslut of predict : %.2f, %.2f", prediction[0], prediction[1]);
    // if(prediction[0] < 0.5 && prediction[0] >= 0.0){
    //   tft.setTextSize(1);
    //   tft.setTextColor(TFT_RED, TFT_BLACK);
    //   tft.setTextPadding(200); 
    //   tft.drawString("high blood sugar", 30, 295, 4); 
    // }
    // else if(prediction[0] > 0.5 && prediction[0] <= 1.0){
    //   tft.setTextSize(1);
    //   tft.setTextColor(TFT_GREEN, TFT_BLACK);
    //   tft.setTextPadding(200); 
    //   tft.drawString("low blood sugar", 30, 295, 4);
    // }
    tft.setTextSize(1);
    tft.setTextColor(TFT_WHITE, TFT_BLACK);
    tft.setTextPadding(75);
    tft.drawString("PPG wave", 5, 0, 2);

    tft.fillRect(DISP_X + 1, DISP_Y + 1, DISP_W - 2, DISP_H - 2, TFT_BLACK);
    tft.drawRect(DISP_X, DISP_Y, DISP_W, DISP_H, TFT_WHITE);
    displayWaveform(PPG,times, DISP_X, DISP_Y, DISP_W, DISP_H - 1);

    PPGdetect++;
  }

  if(reset % 2 == 1){
    ESP.restart();
    reset++;
  }
}

//------------------------------------------------------------------------------------------
//NSP32's ISR
void PinReadyTriggerISR() {
  // make sure to call this function when receiving a ready trigger from NSP32
  nsp32.OnPinReadyTriggered();
}
//------------------------------------------------------------------------------------------
//畫按鍵
void drawKeypad(){
  // Draw the keys
  for (uint8_t row = 0; row < 1; row++) {
    for (uint8_t col = 0; col < 3; col++) {
      uint8_t b = col + row * 3;

      tft.setFreeFont(LABEL1_FONT);

      key[b].initButton(&tft, KEY_X + col * (KEY_W + KEY_SPACING_X),
                        KEY_Y + row * (KEY_H + KEY_SPACING_Y), // x, y, w, h, outline, fill, text
                        KEY_W, KEY_H, TFT_WHITE, keyColor[b], TFT_WHITE,
                        keyLabel[b], KEY_TEXTSIZE);
      key[b].drawButton();
    }
  }
}

//------------------------------------------------------------------------------------------
//校正螢幕範圍
void touch_calibrate(){
  uint16_t calData[5];
  uint8_t calDataOK = 0;

  // check file system exists
  if (!SPIFFS.begin()) {
    Serial.println("Formating file system");
    SPIFFS.format();
    SPIFFS.begin();
  }

  // check if calibration file exists and size is correct
  if (SPIFFS.exists(CALIBRATION_FILE)) {
    if (REPEAT_CAL)
    {
      // Delete if we want to re-calibrate
      SPIFFS.remove(CALIBRATION_FILE);
    }
    else
    {
      File f = SPIFFS.open(CALIBRATION_FILE, "r");
      if (f) {
        if (f.readBytes((char *)calData, 14) == 14)
          calDataOK = 1;
        f.close();
      }
    }
  }

  if (calDataOK && !REPEAT_CAL) {
    // calibration data valid
    tft.setTouch(calData);
  } else {
    // data not valid so recalibrate
    tft.fillScreen(TFT_BLACK);
    tft.setCursor(20, 0);
    tft.setTextFont(2);
    tft.setTextSize(1);
    tft.setTextColor(TFT_WHITE, TFT_BLACK);

    tft.println("Touch corners as indicated");

    tft.setTextFont(1);
    tft.println();

    if (REPEAT_CAL) {
      tft.setTextColor(TFT_RED, TFT_BLACK);
      tft.println("Set REPEAT_CAL to false to stop this running again!");
    }

    tft.calibrateTouch(calData, TFT_MAGENTA, TFT_BLACK, 15);

    tft.setTextColor(TFT_GREEN, TFT_BLACK);
    tft.println("Calibration complete!");

    // store data
    File f = SPIFFS.open(CALIBRATION_FILE, "w");
    if (f) {
      f.write((const unsigned char *)calData, 14);
      f.close();
    }
  }
}


//------------------------------------------------------------------------------------------
//TCA6507 LED Driver
int8_t TCA_writeBytes(uint8_t devAddr, uint8_t regAddr, uint8_t length, uint8_t *data){
  uint8_t status = 0;

  Wire.beginTransmission(devAddr);
  Wire.write(regAddr);

  for (uint8_t i = 0; i < length; i++) {
    Wire.write((uint8_t) data[i]);
  }

  status = Wire.endTransmission();
  return status == 0;
}

int8_t TCA_writeByte(uint8_t devAddr, uint8_t regAddr, uint8_t data){
  return TCA_writeBytes(devAddr, regAddr, 1, &data);
}

int8_t TCA_read(uint8_t devAddr, uint8_t regAddr, uint8_t length, uint8_t *data, uint16_t timeout){

  int8_t count = 0;
  uint32_t t1 = millis();

  for (uint8_t k = 0; k < length; k += std::min<int>(length, BUFFER_LENGTH)) {
    Wire.beginTransmission(devAddr);
    Wire.write(regAddr);
    Wire.endTransmission();
    
    Wire.requestFrom(devAddr, (uint8_t) std::min<int> (length - k, BUFFER_LENGTH));

    for (; Wire.available() && (timeout == 0 || millis() - t1 < timeout); count++) {
        data[count] = Wire.read();
    }
  }

  if (timeout > 0 && millis() - t1 >= timeout && count < length) count = -1; // timeout
  
  return count;
}

bool TCA_isConnected(){
  uint8_t result = 0;

  return ( TCA_read(device_address, SELECT0, 1, &result, 0) == 1);
}
//SD Function-------------------------------------------------------------------------------
void listDir(fs::FS &fs, const char * dirname, uint8_t levels){
  Serial.printf("Listing directory: %s\n", dirname);

  File root = fs.open(dirname);
  if(!root){
    Serial.println("Failed to open directory");
    return;
  }
  if(!root.isDirectory()){
    Serial.println("Not a directory");
    return;
  }

  File file = root.openNextFile();
  while(file){
    if(file.isDirectory()){
      Serial.print("  DIR : ");
      Serial.println(file.name());
      if(levels){
        listDir(fs, file.name(), levels -1);
      }
    } else {
      Serial.print("  FILE: ");
      Serial.print(file.name());
      Serial.print("  SIZE: ");
      Serial.println(file.size());
    }
    file = root.openNextFile();
  }
}

void createDir(fs::FS &fs, const char * path){
  Serial.printf("Creating Dir: %s\n", path);
  if(fs.mkdir(path)){
    Serial.println("Dir created");
  } else {
    Serial.println("mkdir failed");
  }
}

void removeDir(fs::FS &fs, const char * path){
  Serial.printf("Removing Dir: %s\n", path);
  if(fs.rmdir(path)){
    Serial.println("Dir removed");
  } else {
    Serial.println("rmdir failed");
  }
}

void readFile(fs::FS &fs, String path){
  Serial.printf("Reading file: %s\n", path);

  File file = fs.open(path);
  if(!file){
    Serial.println("Failed to open file for reading");
    return;
  }

  Serial.print("Read from file: ");
  while(file.available()){
    Serial.write(file.read());
  }
  file.close();
}

void writeFile(fs::FS &fs, String path, float message[] , int messageSize){
  Serial.printf("Writing file: %s\n", path);

  File file = fs.open(path, FILE_WRITE);
  if(!file){
    Serial.println("Failed to open file for writing");
    return;
  }
  for (int i = 0; i < messageSize; i++) {
    file.printf("%.10f",message[i]);
    if (i < messageSize-1) {
        file.print(",");
    }
  }
  file.println();
  file.close();
}

void appendFileAWPPG(fs::FS &fs, String path, float message[] , int messageSize){
  Serial.printf("Appending to file: %s\n", path);

  File file = fs.open(path, FILE_APPEND);
  if(!file){
    Serial.println("Failed to open file for writing");
    return;
  }
  for (int i = 0; i < messageSize; i++) {
    file.printf("%.10f",message[i]);
    if (i < messageSize-1) {
        file.print(",");
    }
  }
  file.print(",");
  file.print(Time);
  file.println();
  file.close();
}
void appendFilePPG(fs::FS &fs, String path, float message[] , int messageSize){
  Serial.printf("Appending to file: %s\n", path);

  File file = fs.open(path, FILE_APPEND);
  if(!file){
    Serial.println("Failed to open file for writing");
    return;
  }
  for (int i = 0; i < messageSize; i++) {
    file.printf("%.10f",message[i]);
    if (i < messageSize-1) {
        file.print(",");
    }
  }
  file.println();
  file.close();
}
void appendFilePPGtimes(fs::FS &fs, String path, unsigned long message[] , int messageSize){
  Serial.printf("Appending to file: %s\n", path);

  File file = fs.open(path, FILE_APPEND);
  if(!file){
    Serial.println("Failed to open file for writing");
    return;
  }
  for (int i = 0; i < messageSize; i++) {
    file.print(message[i]);
    if (i < messageSize-1) {
        file.print(",");
    }
  }
  file.println();
  file.close();
}

void renameFile(fs::FS &fs, String path1, String path2){
  Serial.printf("Renaming file %s to %s\n", path1, path2);
  if (fs.rename(path1, path2)) {
    Serial.println("File renamed");
  } else {
    Serial.println("Rename failed");
  }
}

void deleteFile(fs::FS &fs, String path){
  Serial.printf("Deleting file: %s\n", path);
  if(fs.remove(path)){
    Serial.println("File deleted");
  } else {
    Serial.println("Delete failed");
  }
}

void testFileIO(fs::FS &fs, String path){
  File file = fs.open(path);
  static uint8_t buf[512];
  size_t len = 0;
  uint32_t start = millis();
  uint32_t end = start;
  if(file){
    len = file.size();
    size_t flen = len;
    start = millis();
    while(len){
      size_t toRead = len;
      if(toRead > 512){
        toRead = 512;
      }
      file.read(buf, toRead);
      len -= toRead;
    }
    end = millis() - start;
    Serial.printf("%u bytes read for %u ms\n", flen, end);
    file.close();
  } else {
    Serial.println("Failed to open file for reading");
  }


  file = fs.open(path, FILE_WRITE);
  if(!file){
    Serial.println("Failed to open file for writing");
    return;
  }

  size_t i;
  start = millis();
  for(i=0; i<2048; i++){
    file.write(buf, 512);
  }
  end = millis() - start;
  Serial.printf("%u bytes written for %u ms\n", 2048 * 512, end);
  file.close();
}
int filenumber() {
  int number =0;
  String searchName = "/AWPPG" + String(number) + fileformat;
  while (true){
    if (!SD.exists(searchName))
    {
      Serial.print("filenumber ");
      Serial.println(number);
      break;
    }
    number++;
    searchName = "/AWPPG" + String(number) + fileformat;
  }
  return number;
}

//顯示Waveform--------------------------------------------------------------------------------
void displayWaveform(float Wave[], int timePoints, int screen_x, int screen_y, int screen_w, int screen_h) {
  float Max = *std::max_element(Wave, Wave + timePoints);
  float Min = *std::min_element(Wave, Wave + timePoints);
  for (int i = 0; i < timePoints; i++){
    Wave[i] = (Wave[i] - Min)/(Max - Min);
  }
  
  // 假設PPG的數據範圍是 0 到 1
  float minVal = 0.0;
  float maxVal = 1.0;

  // 計算X軸和Y軸映射的縮放比例
  float xScale = (float)screen_w / (float)timePoints;  // 將數據點映射到寬度
  float yScale = screen_h / (maxVal - minVal);  // 將數據範圍映射到高度

  // 繪製波形
  for (int i = 1; i < timePoints; i++) {
    // 當前點的Y位置
    int y1 = (Wave[i - 1] - minVal) * yScale - screen_y;
    int y2 = (Wave[i] - minVal) * yScale - screen_y;

    // 計算X軸位置
    int x1 = (i - 1) * xScale + screen_x;
    int x2 = i * xScale + screen_x;

    // 畫線連接兩個相鄰的點
    tft.drawLine(x1, screen_h - y1, x2, screen_h - y2, TFT_GREEN);  // 使用綠色畫線
  }
}

